/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFDensityImage.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for the creation and destruction of density images
 *    Ronald Perry and Sarah Frisken
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
#include <stdlib.h>

#include "fs_object.h"
#include "fs_function.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#ifdef FS_EDGE_RENDER

#include "adfinittermsystem.h"

/**
 *-----------------------------------------------------------------------------------
 *    Create a density image of the specified type and size. A variable-length array of
 *    ADF_U32s can be allocated along with the density image to store
 *    application-specific data associated with the density image; appDataLen specifies
 *    the length of this array. A NULL is returned if the request cannot be satisfied.
 *-----------------------------------------------------------------------------------
 */
ADFImage *ADFCreateImage (void *libInst, ADF_U16 type, ADF_U16 w, ADF_U16 h, 
                          ADF_U16 appDataLen)
{
    ADF_U32      bytesPerPixel;
    ADF_U32      imageSizeBytes;
    ADF_U32      appDataSizeBytes;
    ADF_U32      totalSizeBytes;
    ADFImage     *image;


    /**
     *----Determine the total size of the image header, the image data, and the 
     *----variable-length array of application-specific data
     */
    bytesPerPixel    = (type == ADF_IMAGE_TYPE_GRAY ? 1 : 4);
    imageSizeBytes   = ((ADF_U32) w) * ((ADF_U32) h) * bytesPerPixel;
    appDataSizeBytes = ((ADF_U32) appDataLen) * sizeof(ADF_U32);
    totalSizeBytes   = sizeof(ADFImage) + imageSizeBytes + appDataSizeBytes;


    /**
     *----Allocate all the data (header, image, application) in a single contiguous 
     *----block of memory
     */
#ifdef FS_MEM_DBG
    ((FS_STATE *)ADFGetAppInst(libInst))->memdbgid = "ADFImage";
#endif
    image = (ADFImage *) ADF_ALLOC(ADFGetAppInst(libInst), totalSizeBytes);
    if (image == 0) 
    {
        ((FS_STATE *)ADFGetAppInst(libInst))->error = ERR_MALLOC_FAIL;
        return(0);
    }


    /**
     *----Set the elements of the image header
     */
    image->type = type;
    image->w = w;
    image->h = h;
    image->appDataLen = appDataLen;
    image->appData = (ADF_U32 *) (appDataLen == 0 ? (0) : (image + 1));
    image->base = ((ADF_U8 *) (image + 1)) + appDataSizeBytes;


    /**
     *----Return the requested image
     */
    return(image);
}


/**
 *-----------------------------------------------------------------------------------
 *    Destroy the given density image
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFDestroyImage (void *libInst, ADFImage *image)
{
    if (image) ADF_FREE(ADFGetAppInst(libInst), image);
}

/**
 *-----------------------------------------------------------------------------------
 *    END: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#endif
